//+------------------------------------------------------------------+
//|                                                LR model Test.mq5 |
//|                                  Copyright 2024, MetaQuotes Ltd. |
//|                                             https://www.mql5.com |
//+------------------------------------------------------------------+
#property copyright "Copyright 2024, MetaQuotes Ltd."
#property link      "https://www.mql5.com"
#property version   "1.00"

#resource "\\Files\\EURUSD.dailytf.model.onnx" as uchar lr_onnx[]

#include <Linear Regression.mqh>
#include <pandas.mqh>
#include <ta-lib.mqh>

CLinearRegression lr;

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
#include <Trade\Trade.mqh>
#include <Trade\PositionInfo.mqh>

CTrade m_trade;
CPositionInfo m_position;

input int magic_number = 27012025;
input int slippage = 100;
input int atr_period = 14;
input string symbol_ = "EURUSD";

vector open_, 
       high_, 
       low_, 
       close_;
       
vector pct_change;
vector var_5;
vector atr;

int OldNumBars = -1;
int atr_handle;
//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit()
  {
//---
   
   if (!lr.Init(lr_onnx))
     return INIT_FAILED;   
//---
   
   m_trade.SetExpertMagicNumber(magic_number);
   m_trade.SetTypeFillingBySymbol(symbol_);
   m_trade.SetMarginMode();
   m_trade.SetDeviationInPoints(slippage);

   atr_handle = iATR(symbol_, PERIOD_D1, atr_period);
   
   return(INIT_SUCCEEDED);
  }
//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
  {
//---
   
  }
//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick()
  {
    MqlTick ticks;
    SymbolInfoTick(symbol_, ticks);
    
     if (isNewBar())
       {
           vector x = GetData();   
           double predicted_close = lr.predict(x);
           double min_volume = SymbolInfoDouble(symbol_, SYMBOL_VOLUME_MIN);
           
           vector atr_value;
           atr_value.CopyIndicatorBuffer(atr_handle, 0, 1, 1);
           
           double stoploss, takeprofit;
           
           if (predicted_close>ticks.bid && !PosExists(POSITION_TYPE_BUY))
             {
               stoploss = ticks.ask - atr_value[0];
               takeprofit = ticks.ask + atr_value[0];
           
               m_trade.Buy(min_volume, symbol_, ticks.ask, stoploss, takeprofit);
             }
             
           if (predicted_close<ticks.ask && !PosExists(POSITION_TYPE_SELL))
             {
               stoploss = ticks.bid + atr_value[0];
               takeprofit = ticks.bid - atr_value[0];
           
               m_trade.Sell(min_volume, symbol_, ticks.bid, stoploss, takeprofit);
             }
       }
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool isNewBar()
  {
   int CurrentNumBars = Bars(symbol_, PERIOD_D1);
   if(OldNumBars!=CurrentNumBars)
     {
      OldNumBars = CurrentNumBars;
      return true;
     }
   return false;
  }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
bool PosExists(ENUM_POSITION_TYPE type)
 {
    for (int i=PositionsTotal()-1; i>=0; i--)
      if (m_position.SelectByIndex(i))
         if (m_position.Symbol()==symbol_ && m_position.Magic() == magic_number && m_position.PositionType()==type)
            return (true);
            
    return (false);
 }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
vector GetData(int start_bar=1, int size=30)
 {
    CDataFrame df_;
    
    open_.CopyRates(symbol_, PERIOD_D1, COPY_RATES_OPEN,start_bar, size);
    high_.CopyRates(symbol_, PERIOD_D1, COPY_RATES_HIGH,start_bar, size);
    low_.CopyRates(symbol_, PERIOD_D1, COPY_RATES_LOW,start_bar, size);
    close_.CopyRates(symbol_, PERIOD_D1, COPY_RATES_CLOSE,start_bar, size);
    
    df_.Insert("open",open_);
    df_.Insert("high",high_);
    df_.Insert("low",low_);
    df_.Insert("close",close_);
    
    
    int lags = 5;
    vector lag = {};
    
    for (int i=1; i<=lags; i++)
      {
         lag = df_.Shift("close", i);
         df_.Insert("close lag_"+string(i), lag);
      }
    
    pct_change = df_.Pct_change("close");
    df_.Insert("close pct_change", pct_change);
    
    var_5 = df_.Rolling("close", 5).Var();
    df_.Insert("var close 5 days", var_5);
    
    df_.Insert("open_close",open_-close_);
    df_.Insert("high_low",high_-low_);
    
    df_.Insert("Avg price",(open_+high_+low_+close_)/4);

//---
       
    BB_res_struct bb = CTrendIndicators::BollingerBands(close_,20,0,2.000000); //Calculating the bollinger band indicator
    
    df_.Insert("bb_lower",bb.lower_band); //Inserting lower band values
    df_.Insert("bb_middle",bb.middle_band); //Inserting the middle band values
    df_.Insert("bb_upper",bb.upper_band); //Inserting the upper band values
    
    atr = COscillatorIndicators::ATR(high_,low_,close_,14);  //Calculating the ATR Indicator
    
    df_.Insert("ATR 14",atr); //Inserting the ATR indicator values
    
    MACD_res_struct macd = COscillatorIndicators::MACD(close_,12,26,9); //MACD indicator applied to the closing price
    
    df_.Insert("macd histogram", macd.histogram); //Inserting the MAC historgram values
    df_.Insert("macd main", macd.main); //Inserting the macd main line values 
    df_.Insert("macd signal", macd.signal);  //Inserting the macd signal line values 
    
    
    df_ = df_.Drop(
       //"future_close", 
       "var close 5 days,"+
       "ATR 14,"+
       "close pct_change,"+
       "close lag_4,"+
       "close lag_2,"+
       "open_close,"+
       "high,"+
       "low,"+
       "macd histogram,"+
       "macd signal"   
    );
    
    CDataFrame new_df = df_.Dropnan();
    
    new_df.Head();
    
    return new_df.Loc(-1); //return the latest row
 }
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
